from pyexpat.errors import messages
import uuid
from datetime import timedelta, date # Import date
from django.shortcuts import render, redirect, get_object_or_404
from django.utils import timezone
from django.db.models import Count
from django.contrib.auth.decorators import login_required
from django.views.decorators.http import require_POST
from django.contrib import messages
import random

from .models import (
    User,
    StudentProfile,
    ClassLevel,
    Subject,
    Chapter,
    Question,
    Choice,
    Exam,
    ExamQuestion,
    ExamResult,
    ExamCategory,
    ExamResultDetails, # Import new model
)
from subscription.models import (
    AnnualSubscription,
    AnnualSubscriptionDetails,
    SubjectBasedSubscription,
    SubjectBasedSubscriptionDetails,
    BundleSubscription,
    BundleSubscriptionDetails,
)


# =========================================
# HOME & DASHBOARD
# =========================================

def home(request):
    # Home page doesn't require login
    user = request.user
    student = None
    
    if user.is_authenticated:
        student = StudentProfile.objects.filter(user=user).first()  # safely get 1 object or None

    # Get bundles: prioritize student's class, fallback to first available class level
    bundles = BundleSubscriptionDetails.objects.none()
    
    if student and student.current_class:
        bundles = BundleSubscriptionDetails.objects.filter(class_level=student.current_class)[:3]
    else:
        # For unauthenticated or users without class: show first 3 bundles from any available class
        bundles = BundleSubscriptionDetails.objects.all().order_by('class_level')[:3]
        
    context = {
        'bundles': bundles,
    }
    return render(request, 'web/home/index.html', context)


@login_required(login_url='login_user')
def DashboardPage(request):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to access your dashboard.")
        return redirect('login_user')
    
    try:
        user = request.user 
        student_profile = get_object_or_404(StudentProfile, user=request.user)
        
        # Fetch exams that have a result, ordered by completion date
        all_exam_results = ExamResult.objects.filter(
            exam__student=student_profile
        ).select_related('exam').order_by('-completed_at')[:3]

        bundle_subscription = BundleSubscription.objects.filter(student=student_profile, is_active=True)[:4]

        bundle_subscription_count = bundle_subscription.count()

        empty_subscription = False
        if bundle_subscription_count == 0:
            empty_subscription = True

        context = {
            'user': user,
            'all_exam_results': all_exam_results,
            'bundle_subscription': bundle_subscription,
            'empty_subscription': empty_subscription,
        }
        return render(request, 'dashboard/dashboard/dashboard.html', context)
    
    except Exception as e:
        messages.error(request, f"Error accessing dashboard: {str(e)}")
        return redirect('app:home')


# =========================================
# SUBSCRIPTION PAGES
# =========================================
def subscriptionPage(request):
    # Subscription info pages don't require login
    return render(request, 'web/subscription/subscriptions.html')


def AnnualSubscriptionPage(request):
    # Subscription info pages don't require login
    annual_subscription = AnnualSubscriptionDetails.objects.prefetch_related('keys').all()
    return render(request, 'web/subscription/annual_subscription.html', {
        'annual_subscription': annual_subscription
    })


def SubjectSubscriptionPage(request):
    # Subscription info pages don't require login
    subject_subscription = SubjectBasedSubscriptionDetails.objects.prefetch_related('keys').all()
    return render(request, 'web/subscription/subject_subcription.html', {
        'subject_subscription': subject_subscription
    })


def BundleSubscriptionPage(request):
    # Subscription info pages don't require login
    bundle_subscription = BundleSubscriptionDetails.objects.prefetch_related('subjects').all()
    return render(request, 'web/subscription/bundle_subscription.html', {
        'bundle_subscription': bundle_subscription
    })


# =========================================
# CHECKOUT & PROCESSING
# =========================================
@login_required(login_url='login_user')
def checkoutPageAnnual(request, pk):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to complete your purchase.")
        return redirect('login_user') + f'?next={request.path}'
    
    subscription = get_object_or_404(AnnualSubscriptionDetails, pk=pk)
    return render(request, 'web/subscription/payment.html', {
        'subscription': subscription,
        'subscription_type': 'annual',
        'current_year': timezone.now().year,
    })


@login_required(login_url='login_user')
def checkoutPageSubjectBased(request, pk):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to complete your purchase.")
        return redirect('login_user') + f'?next={request.path}'
    
    subscription = get_object_or_404(SubjectBasedSubscriptionDetails, pk=pk)
    return render(request, 'web/subscription/payment.html', {
        'subscription': subscription,
        'subscription_type': 'subject_based',
        'current_year': timezone.now().year,
    })


@login_required(login_url='login_user')
def checkoutPageBundle(request, pk):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to complete your purchase.")
        return redirect('login_user') + f'?next={request.path}'
    
    subscription = get_object_or_404(BundleSubscriptionDetails, pk=pk)
    return render(request, 'web/subscription/payment.html', {
        'subscription': subscription,
        'subscription_type': 'bundle',
        'current_year': timezone.now().year,
    })


@login_required(login_url='login_user')
def process_subscription(request):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to complete your purchase.")
        return redirect('login_user')
        
    if request.method == 'POST':
        try:
            form_data = {
                'subscription_type': request.POST.get('subscription_type'),
                'subscription_id': request.POST.get('subscription_id'),
                'first_name': request.POST.get('first_name'),
                'last_name': request.POST.get('last_name'),
                'email': request.POST.get('email'),
                'phone': request.POST.get('phone'),
                'address': request.POST.get('address'),
                'city': request.POST.get('city'),
                'payment_method': request.POST.get('payment_method')
            }
            if not all(form_data.values()):
                messages.error(request, "Please fill in all required fields.")
                return redirect(request.META.get('HTTP_REFERER'))

            user = request.user
            user.first_name = form_data['first_name']
            user.last_name = form_data['last_name']
            user.email = form_data['email']
            user.phone_number = form_data['phone']
            user.save()

            student_profile, _ = StudentProfile.objects.get_or_create(user=user)
            student_profile.school_name = f"{form_data['address']}, {form_data['city']}"
            student_profile.save()

            subscription = create_subscription(
                form_data['subscription_type'],
                form_data['subscription_id'],
                student_profile
            )

            if subscription:
                messages.success(request, "Subscription purchased successfully!")
                return redirect('app:dashboard')
            else:
                messages.error(request, "Failed to create subscription.")
                return redirect('app:subscription_page')

        except Exception as e:
            messages.error(request, f"Error processing subscription: {str(e)}")
            return redirect(request.META.get('HTTP_REFERER'))
    return redirect('app:subscription_page')


def create_subscription(subscription_type, subscription_id, student_profile):
    today = timezone.now().date()
    if subscription_type == 'annual':
        details = get_object_or_404(AnnualSubscriptionDetails, id=subscription_id)
        expiration_date = date(today.year + 1, today.month, today.day)
        return AnnualSubscription.objects.create(
            annual_subscription_details=details,
            student=student_profile,
            class_level=details.class_level,
            start_date=today,
            end_date=expiration_date,
            is_active=True
        )
    elif subscription_type == 'subject_based':
        details = get_object_or_404(SubjectBasedSubscriptionDetails, id=subscription_id)
        expiration_date = date(today.year, 12, 31)
        subscription = SubjectBasedSubscription.objects.create(
            subject_based_subscription_details=details,
            student=student_profile,
            class_level=details.class_level,
            start_date=today,
            end_date=expiration_date,
            is_active=True
        )
        subscription.subjects.add(details.subject)
        return subscription
    elif subscription_type == 'bundle':
        details = get_object_or_404(BundleSubscriptionDetails, id=subscription_id)
        expiration_date = date(today.year, 12, 31)
        subscription = BundleSubscription.objects.create(
            bundle_subscription_details=details,
            student=student_profile,
            class_level=details.class_level,
            start_date=today,
            end_date=expiration_date,
            is_active=True
        )
        subscription.subjects.set(details.subjects.all())
        return subscription
    return None


# =========================================
# MCQ FLOW VIEWS
# =========================================

@login_required(login_url='login_user')
def MCQ_Page(request):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to access practice tests.")
        return redirect('login_user')
    
    try:
        # 1. Get student profile
        student_profile = StudentProfile.objects.get(user=request.user)
        selected_subscription = request.GET.get('subscription', None)

        # 2. Get all active subscriptions with related data
        active_subs = {
            'annual': AnnualSubscription.objects.filter(
                student=student_profile,
                is_active=True,
                end_date__gte=timezone.now().date()
            ).select_related('annual_subscription_details__subject', 'class_level'),

            'subject_based': SubjectBasedSubscription.objects.filter(
                student=student_profile,
                is_active=True,
                end_date__gte=timezone.now().date()
            ).prefetch_related('subjects', 'class_level'),

            'bundle': BundleSubscription.objects.filter(
                student=student_profile,
                is_active=True,
                end_date__gte=timezone.now().date()
            ).prefetch_related('subjects', 'class_level', 'bundle_subscription_details')
        }

        # 3. Collect subjects with their subscription info
        subscribed_subjects = []
        subscription_titles = set()
        
        # Define color options (text and border colors)
        color_options = [
            {'text': 'text-red-600', 'border': 'border-red-500'},
            {'text': 'text-blue-600', 'border': 'border-blue-500'},
            {'text': 'text-green-600', 'border': 'border-green-500'},
            {'text': 'text-yellow-600', 'border': 'border-yellow-500'},
            {'text': 'text-purple-600', 'border': 'border-purple-500'},
            {'text': 'text-pink-600', 'border': 'border-pink-500'},
            {'text': 'text-indigo-600', 'border': 'border-indigo-500'},
            {'text': 'text-teal-600', 'border': 'border-teal-500'},
            {'text': 'text-orange-600', 'border': 'border-orange-500'},
            {'text': 'text-amber-600', 'border': 'border-amber-500'},
        ]

        # Track colors assigned to each subscription title
        title_colors = {}

        # Process annual subscriptions
        for sub in active_subs['annual']:
            if sub.annual_subscription_details and sub.annual_subscription_details.subject:
                title = sub.annual_subscription_details.title or "Annual Subscription"
                if title not in title_colors:
                    title_colors[title] = random.choice(color_options)
                    subscription_titles.add(title)
                
                if not selected_subscription or selected_subscription == title:
                    subscribed_subjects.append({
                        'subject': sub.annual_subscription_details.subject,
                        'subscription_title': title,
                        'type': 'annual',
                        'class_level': sub.class_level,
                        'color': title_colors[title]
                    })
        
        # Process subject-based subscriptions
        for sub in active_subs['subject_based']:
            if sub.subject_based_subscription_details:
                title = sub.subject_based_subscription_details.title or "Subject Subscription"
                if title not in title_colors:
                    title_colors[title] = random.choice(color_options)
                    subscription_titles.add(title)
                
                if not selected_subscription or selected_subscription == title:
                    for subject in sub.subjects.all():
                        subscribed_subjects.append({
                            'subject': subject,
                            'subscription_title': title,
                            'type': 'subject_based',
                            'class_level': sub.class_level,
                            'color': title_colors[title]
                        })
        
        # Process bundle subscriptions
        for sub in active_subs['bundle']:
            if sub.bundle_subscription_details:
                title = sub.bundle_subscription_details.title or "Bundle Subscription"
                if title not in title_colors:
                    title_colors[title] = random.choice(color_options)
                    subscription_titles.add(title)
                
                if not selected_subscription or selected_subscription == title:
                    for subject in sub.subjects.all():
                        subscribed_subjects.append({
                            'subject': subject,
                            'subscription_title': title,
                            'type': 'bundle',
                            'class_level': sub.class_level,
                            'color': title_colors[title]
                        })

        # Get unique class levels
        class_levels = ClassLevel.objects.filter(
            id__in={s['class_level'].id for s in subscribed_subjects if s['class_level']}
        ).order_by('name')

        context = {
            'subscribed_subjects': subscribed_subjects,
            'subscription_titles': sorted(subscription_titles),
            'selected_subscription': selected_subscription,
            'class_levels': class_levels,
            'categories': ExamCategory.objects.all(),
            'entries': Exam.objects.filter(student=student_profile)
        }

        return render(request, 'dashboard/mcq_test/mcq_page.html', context)

    except StudentProfile.DoesNotExist:
        messages.error(request, 'Student profile not found. Please complete your profile.')
        return redirect('app:dashboard')
    except Exception as e:
        messages.error(request, f'Error loading MCQ page: {str(e)}')
        return redirect('app:dashboard')


@login_required(login_url='login_user')
def mcq_chapter_page(request, subject_id):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to access this page.")
        return redirect('login_user')
    
    subject = get_object_or_404(Subject, id=subject_id)
    
    chapters = (
        Chapter.objects
        .filter(subject=subject)
        .annotate(question_count=Count('questions'))
    )

    subject_question_count = Question.objects.filter(subject=subject).count()

    context = {
        'subject': subject,
        'chapters': chapters,
        'subject_question_count': subject_question_count,
    }
    return render(request, 'dashboard/mcq_test/choose_chapter.html', context)


@login_required(login_url='login_user')
def preview_page(request):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to access this page.")
        return redirect('login_user')
        
    try:
        subject_id = request.GET.get('subject')
        chapters_param = request.GET.getlist('chapters')
        
        if not subject_id or not chapters_param:
            messages.warning(request, "Please select both subject and chapters.")
            return redirect('app:mcq_test')

        subject = get_object_or_404(Subject, id=subject_id)
        selected_chapters = Chapter.objects.filter(id__in=chapters_param)
        
        if not selected_chapters.exists():
            messages.warning(request, "No chapters selected.")
            return redirect('app:mcq_chapter_page', subject_id=subject_id)

        total_questions = Question.objects.filter(
            chapter_id__in=chapters_param
        ).count()

        context = {
            'subject': subject,
            'selected_chapters': selected_chapters,
            'total_questions': total_questions,
            'subject_id': subject_id,
            'chapter_ids_str': ','.join(chapters_param),
            'title': f"{subject.name} - {selected_chapters.first().title if selected_chapters.first() else 'Custom'} Exam"
        }
        return render(request, 'dashboard/mcq_test/final_preview.html', context)
    
    except Exception as e:
        messages.error(request, f"Error loading preview: {str(e)}")
        return redirect('app:mcq_test')
    

@login_required(login_url='login_user')
@require_POST
def start_exam(request):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to start an exam.")
        return redirect('login_user')
        
    user_profile = get_object_or_404(StudentProfile, user=request.user)

    subject_id = request.POST.get('subject_id')
    chapter_ids_str = request.POST.get('chapter_ids', '')
    num_questions = int(request.POST.get('num_questions', 0))
    duration_minutes = int(request.POST.get('duration_minutes', 0))
    exam_title = request.POST.get('exam_title', 'Custom Exam')

    if not subject_id or not chapter_ids_str or num_questions <= 0 or duration_minutes <= 0:
        messages.error(request, "Invalid exam parameters.")
        return redirect('app:mcq_test')

    subject = get_object_or_404(Subject, id=subject_id)
    chapter_ids = [c for c in chapter_ids_str.split(',') if c]

    exam = Exam.objects.create(
        title=exam_title,
        student=user_profile,
        subject=subject,
        class_level=subject.class_level,
        exam_category=None,
        duration_minutes=duration_minutes,
        is_custom=True
    )

    questions_qs = Question.objects.filter(chapter_id__in=chapter_ids)
    selected_questions = questions_qs.order_by('?')[:num_questions]

    for q in selected_questions:
        ExamQuestion.objects.create(exam=exam, question=q)

    return redirect('app:exam_page', exam_id=exam.id)


@login_required(login_url='login_user')
def exam_page(request, exam_id):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to access this exam.")
        return redirect('login_user')
        
    exam = get_object_or_404(Exam, id=exam_id, student__user=request.user)
    
    # Ensure exam is not already finished
    if hasattr(exam, 'result'):
        messages.info(request, "This exam has already been completed.")
        return redirect('app:exam_results', exam_id=exam.id)

    exam_questions = exam.exam_questions.select_related('question').all()

    context = {
        'exam': exam,
        'exam_questions': exam_questions,
    }
    return render(request, 'dashboard/mcq_test/exam_page.html', context)


@login_required(login_url='login_user')
@require_POST
def finish_exam(request, exam_id):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to submit an exam.")
        return redirect('login_user')
        
    exam = get_object_or_404(Exam, id=exam_id, student__user=request.user)

    # If exam result already exists, redirect to avoid re-grading
    if hasattr(exam, 'result'):
        messages.info(request, "This exam has already been submitted.")
        return redirect('app:exam_results', exam_id=exam.id)

    correct_count = 0
    wrong_count = 0
    total_questions_answered = 0
    exam_result_details_list = []

    exam_start_time = exam.created_at
    exam_duration = timedelta(minutes=exam.duration_minutes)
    exam_end_time = exam_start_time + exam_duration
    
    now = timezone.now()
    time_taken_seconds = min(int((now - exam_start_time).total_seconds()), exam_duration.total_seconds())

    for eq in exam.exam_questions.select_related('question').all():
        field_name = f"q_{eq.question.id}"
        selected_choice_id = request.POST.get(field_name)
        
        selected_choice = None
        is_correct = False

        if selected_choice_id:
            total_questions_answered += 1
            try:
                choice = Choice.objects.get(id=selected_choice_id, question=eq.question)
                selected_choice = choice
                if choice.is_correct:
                    correct_count += 1
                    is_correct = True
                else:
                    wrong_count += 1
            except Choice.DoesNotExist:
                wrong_count += 1
        else:
            wrong_count += 1

        correct_choice = eq.question.choices.filter(is_correct=True).first()
        exam_result_details_list.append(
            ExamResultDetails(
                exam_question=eq,
                selected_choice=selected_choice,
                is_correct=is_correct,
                correct_choice=correct_choice
            )
        )

    score = correct_count
    
    exam_result = ExamResult.objects.create(
        exam=exam,
        score=score,
        correct_answers=correct_count,
        wrong_answers=wrong_count,
        time_taken_seconds=int(time_taken_seconds)
    )
    
    for detail in exam_result_details_list:
        detail.exam_result = exam_result
        detail.save()

    messages.success(request, "Exam submitted successfully!")
    return redirect('app:exam_results', exam_id=exam.id)


@login_required(login_url='login_user')
def exam_results(request, exam_id):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to view exam results.")
        return redirect('login_user')
        
    exam = get_object_or_404(Exam, id=exam_id, student__user=request.user)
    result = get_object_or_404(ExamResult, exam=exam)

    exam_result_details = ExamResultDetails.objects.filter(exam_result=result).select_related(
        'exam_question__question', 'selected_choice', 'correct_choice'
    ).order_by('exam_question__question__text')

    # Calculate statistics
    total_questions = exam.exam_questions.count()
    correct_answers = result.correct_answers
    incorrect_answers = result.wrong_answers
    unanswered = total_questions - (correct_answers + incorrect_answers)
    
    # Calculate percentage score
    score = (correct_answers / total_questions * 100) if total_questions > 0 else 0
    
    # Format time taken
    time_seconds = result.time_taken_seconds
    minutes = time_seconds // 60
    seconds = time_seconds % 60
    time_taken = f"{minutes}m {seconds}s"

    context = {
        'exam': exam,
        'result': result,
        'exam_result_details': exam_result_details,
        'total_questions': total_questions,
        'correct_answers': correct_answers,
        'incorrect_answers': incorrect_answers,
        'unanswered': unanswered,
        'score': score,
        'time_taken': time_taken,
    }
    return render(request, 'dashboard/mcq_test/exam_results.html', context)


@login_required(login_url='login_user')
def retake_exam(request, exam_id):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to retake an exam.")
        return redirect('login_user')
        
    original_exam = get_object_or_404(Exam, id=exam_id, student__user=request.user)
    user_profile = original_exam.student

    new_exam = Exam.objects.create(
        title=f"Retake: {original_exam.title}",
        student=user_profile,
        subject=original_exam.subject,
        class_level=original_exam.class_level,
        exam_category=original_exam.exam_category,
        duration_minutes=original_exam.duration_minutes,
        is_custom=original_exam.is_custom
    )

    original_exam_questions_ids = original_exam.exam_questions.values_list('question_id', flat=True)
    shuffled_questions = list(Question.objects.filter(id__in=original_exam_questions_ids).order_by('?'))

    for q in shuffled_questions:
        ExamQuestion.objects.create(exam=new_exam, question=q)

    messages.success(request, "New exam created for retake!")
    return redirect('app:exam_page', exam_id=new_exam.id)


@login_required(login_url='login_user')
def TotalResultPage(request):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to view your results.")
        return redirect('login_user')
        
    student_profile = get_object_or_404(StudentProfile, user=request.user)
    
    all_exam_results = ExamResult.objects.filter(
        exam__student=student_profile
    ).select_related('exam').order_by('-completed_at')

    context = {
        'all_exam_results': all_exam_results
    }
    return render(request, 'dashboard/mcq_test/total_results_page.html', context)

# =========================================
# SETTINGS & SUBSCRIPTIONS PAGE
# =========================================
@login_required(login_url='login_user')
def SubscriptionPage(request):
    if not request.user.is_authenticated:
        messages.warning(request, "Please log in to view your subscriptions.")
        return redirect('login_user')
        
    user = request.user
    student_profile = StudentProfile.objects.get(user=user)
    annual_subscription = AnnualSubscription.objects.filter(student=student_profile, is_active=True)
    subject_subscription = SubjectBasedSubscription.objects.filter(student=student_profile, is_active=True)
    bundle_subscription = BundleSubscription.objects.filter(student=student_profile, is_active=True)

    annual_subscription_count = annual_subscription.count()
    subject_subscription_count = subject_subscription.count()
    bundle_subscription_count = bundle_subscription.count()

    empty_subscription = False
    if annual_subscription_count == 0 and subject_subscription_count == 0 and bundle_subscription_count == 0:
        empty_subscription = True

    context = {
        'annual_subscription': annual_subscription,
        'subject_subscription': subject_subscription,
        'bundle_subscription': bundle_subscription,
        'empty_subscription': empty_subscription,
    }
    return render(request, 'dashboard/subscription/subscription.html', context)















def bulk_add_questions(request):
    if request.method == 'POST':
        class_level_id = request.POST.get('class_level')
        subject_id = request.POST.get('subject')
        chapter_id = request.POST.get('chapter')

        index = 0
        while True:
            question_text = request.POST.get(f'questions[{index}][text]')
            if not question_text:
                break

            question = Question.objects.create(
                text=question_text,
                question_type='MCQ',
                subject_id=subject_id,
                chapter_id=chapter_id,
                difficulty=1
            )

            correct_index = request.POST.get(f'questions[{index}][correct]')
            explanation = request.POST.get(f'questions[{index}][explanation]', '')

            for i in range(4):
                choice_text = request.POST.get(f'questions[{index}][choices][{i}]', '')
                if choice_text:
                    is_correct = (str(i) == correct_index)
                    Choice.objects.create(
                        question=question,
                        text=choice_text,
                        is_correct=is_correct,
                        explanation=explanation if is_correct else ''
                    )
            index += 1

        messages.success(request, 'Questions added successfully!')
        return redirect('app:bulk_add_questions')

    context = {
        'class_levels': ClassLevel.objects.all(),
        'subjects': Subject.objects.all(),
        'chapters': Chapter.objects.all(),
    }
    return render(request, 'custom_add_questions.html', context)


# =========================================
# QUESTION BANK VIEWS
# =========================================

@login_required(login_url='login_user')
def question_bank(request):
    """Question bank main page showing available subjects"""
    student = StudentProfile.objects.get(user=request.user)
    subjects = Subject.objects.all()
    
    context = {
        'subjects': subjects,
        'student': student,
    }
    return render(request, 'dashboard/question_bank/bank.html', context)


@login_required(login_url='login_user')
def subject_questions(request, subject_id):
    """Show all chapters and questions for a subject"""
    subject = get_object_or_404(Subject, id=subject_id)
    chapters = Chapter.objects.filter(subject=subject)
    student = StudentProfile.objects.get(user=request.user)
    
    context = {
        'subject': subject,
        'chapters': chapters,
        'student': student,
    }
    return render(request, 'dashboard/question_bank/subject_questions.html', context)


@login_required(login_url='login_user')
def chapter_questions(request, chapter_id):
    """Show all questions in a specific chapter"""
    chapter = get_object_or_404(Chapter, id=chapter_id)
    questions = Question.objects.filter(chapter=chapter)
    student = StudentProfile.objects.get(user=request.user)
    
    context = {
        'chapter': chapter,
        'subject': chapter.subject,
        'questions': questions,
        'student': student,
    }
    return render(request, 'dashboard/question_bank/subject_questions.html', context)


@login_required(login_url='login_user')
def question_detail(request, question_id):
    """Show detailed view of a single question"""
    question = get_object_or_404(Question, id=question_id)
    chapter = question.chapter
    subject = chapter.subject
    student = StudentProfile.objects.get(user=request.user)
    
    context = {
        'question': question,
        'chapter': chapter,
        'subject': subject,
        'student': student,
    }
    return render(request, 'dashboard/question_bank/question_detail.html', context)