from django.shortcuts import redirect, render, get_object_or_404
from django.http import JsonResponse
from django.views.decorators.http import require_POST
from django.contrib.auth.decorators import login_required
from django.db.models import Count, Q
from django.core.paginator import Paginator
from app.models import ClassLevel, Subject, Chapter
from .models import Doubt, Comment, DoubtImage, DoubtDocument, DoubtLink, Notification, Bookmark
from django.db import transaction
import json
from django.db import transaction
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.contrib.auth.decorators import login_required
from django.db.models import Count, Q





def doubts_home(request):
    try:
        # Get filter parameters
        class_level = request.GET.get('class')
        subject = request.GET.get('subject')
        chapter = request.GET.get('chapter')
        sort_by = request.GET.get('sort', 'newest')
        
        # Filter doubts
        doubts = Doubt.objects.all().select_related(
            'user', 'class_level', 'subject', 'chapter'
        ).prefetch_related(
            'images', 'documents', 'links', 'comments', 'likes', 'bookmarks'
        )
        
        if class_level:
            doubts = doubts.filter(class_level_id=class_level)
        if subject:
            doubts = doubts.filter(subject_id=subject)
        if chapter:
            doubts = doubts.filter(chapter_id=chapter)
        
        # Sort doubts
        if sort_by == 'newest':
            doubts = doubts.order_by('-created_at')
        elif sort_by == 'oldest':
            doubts = doubts.order_by('created_at')
        elif sort_by == 'most_liked':
            doubts = doubts.annotate(like_count=Count('likes')).order_by('-like_count')
        elif sort_by == 'most_commented':
            doubts = doubts.annotate(comment_count=Count('comments')).order_by('-comment_count')
        
        # Pagination with error handling
        paginator = Paginator(doubts, 10)
        page_number = request.GET.get('page')
        
        try:
            page_obj = paginator.page(page_number)
        except PageNotAnInteger:
            page_obj = paginator.page(1)
        except EmptyPage:
            page_obj = paginator.page(paginator.num_pages)
        
        # Prefetch all data for dropdowns
        class_levels = ClassLevel.objects.prefetch_related(
            'subjects__chapters'
        ).all().order_by('order')
        
        context = {
            'doubts': page_obj,
            'class_levels': class_levels,
            'current_filters': {
                'class': class_level,
                'subject': subject,
                'chapter': chapter,
                'sort': sort_by
            }
        }
        return render(request, 'dashboard/doubts/doubts_home.html', context)
    
    except Exception as e:
        return redirect('app:dashboard')





@login_required
def get_subjects(request):
    class_level_id = request.GET.get('class_level_id')
    subjects = Subject.objects.filter(class_level_id=class_level_id).order_by('name')
    return render(request, 'dashboard/doubts/subjects_dropdown.html', {'subjects': subjects})

@login_required
def get_chapters(request):
    subject_id = request.GET.get('subject_id')
    chapters = Chapter.objects.filter(subject_id=subject_id)
    return render(request, 'dashboard/doubts/chapters_dropdown.html', {'chapters': chapters})

@login_required
@require_POST
def post_doubt(request):
    try:
        # Get form data
        post_type = request.POST.get('post_type')
        class_level_id = request.POST.get('class_level')
        subject_id = request.POST.get('subject')
        chapter_id = request.POST.get('chapter') or None
        title = request.POST.get('title', '').strip()
        content = request.POST.get('content', '').strip()
        
        # Validation
        if not title:
            return JsonResponse({'success': False, 'error': 'Title is required'}, status=400)
        if not class_level_id:
            return JsonResponse({'success': False, 'error': 'Class is required'}, status=400)
        if not subject_id:
            return JsonResponse({'success': False, 'error': 'Subject is required'}, status=400)
        
        with transaction.atomic():
            # Create doubt
            doubt = Doubt.objects.create(
                user=request.user,
                post_type=post_type,
                class_level_id=class_level_id,
                subject_id=subject_id,
                chapter_id=chapter_id if chapter_id else None,
                title=title,
                content=content if content else None
            )
            
            # Handle image uploads
            if post_type == 'image':
                images = request.FILES.getlist('images')
                if images:
                    for i, image in enumerate(images):
                        captions = request.POST.getlist('image_captions')
                        caption = captions[i] if i < len(captions) else ''
                        DoubtImage.objects.create(doubt=doubt, image=image, caption=caption)
            
            # Handle document uploads
            elif post_type == 'document':
                documents = request.FILES.getlist('documents')
                if documents:
                    for doc in documents:
                        DoubtDocument.objects.create(doubt=doubt, document=doc)
            
            # Handle links
            elif post_type == 'link':
                links = request.POST.getlist('links')
                descriptions = request.POST.getlist('link_descriptions')
                if links:
                    for url, desc in zip(links, descriptions):
                        if url and url.strip():
                            DoubtLink.objects.create(
                                doubt=doubt, 
                                url=url.strip(), 
                                description=desc.strip() if desc else ''
                            )
        
        return JsonResponse({
            'success': True,
            'doubt_id': str(doubt.id),
            'message': 'Question posted successfully!'
        })
    except Exception as e:
        return JsonResponse({'success': False, 'error': str(e)}, status=400)

@login_required
@require_POST
def like_doubt(request, doubt_id):
    doubt = get_object_or_404(Doubt, id=doubt_id)
    user = request.user
    
    if user in doubt.likes.all():
        doubt.likes.remove(user)
        liked = False
    else:
        doubt.likes.add(user)
        liked = True
        
        # Create notification if it's not the user's own doubt
        if doubt.user != user:
            Notification.objects.create(
                user=doubt.user,
                doubt=doubt,
                is_read=False
            )
    
    return JsonResponse({
        'success': True,
        'liked': liked,
        'total_likes': doubt.likes.count()
    })

@login_required
@require_POST
def bookmark_doubt(request, doubt_id):
    doubt = get_object_or_404(Doubt, id=doubt_id)
    user = request.user
    
    if user in doubt.bookmarks.all():
        doubt.bookmarks.remove(user)
        bookmarked = False
        # Delete bookmark record if exists
        Bookmark.objects.filter(user=user, doubt=doubt).delete()
    else:
        doubt.bookmarks.add(user)
        bookmarked = True
        # Create bookmark record
        Bookmark.objects.create(user=user, doubt=doubt)
    
    return JsonResponse({
        'success': True,
        'bookmarked': bookmarked,
        'total_bookmarks': doubt.bookmarks.count()
    })

@login_required
@require_POST
def post_comment(request, doubt_id):
    doubt = get_object_or_404(Doubt, id=doubt_id)
    content = request.POST.get('content')
    
    if not content:
        return JsonResponse({'success': False, 'error': 'Comment cannot be empty'}, status=400)
    
    try:
        comment = Comment.objects.create(
            doubt=doubt,
            user=request.user,
            content=content
        )
        
        # Create notification if it's not the user's own doubt
        if doubt.user != request.user:
            Notification.objects.create(
                user=doubt.user,
                doubt=doubt,
                comment=comment,
                is_read=False
            )
        
        return JsonResponse({
            'success': True,
            'comment_id': comment.id,
            'user_name': request.user.username,
            'user_profile_pic': request.user.profile_picture.url if request.user.profile_picture else '',
            'content': comment.content,
            'created_at': comment.created_at.strftime('%b %d, %Y %I:%M %p')
        })
    except Exception as e:
        return JsonResponse({'success': False, 'error': str(e)}, status=400)

@login_required
def my_doubts(request):
    doubts = Doubt.objects.filter(user=request.user).order_by('-created_at')
    paginator = Paginator(doubts, 10)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    return render(request, 'dashboard/doubts/my_doubts.html', {'doubts': page_obj})

@login_required
def my_bookmarks(request):
    bookmarks = Bookmark.objects.filter(user=request.user).select_related('doubt')
    doubt_ids = [bm.doubt.id for bm in bookmarks]
    doubts = Doubt.objects.filter(id__in=doubt_ids).order_by('-created_at')
    
    paginator = Paginator(doubts, 10)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    return render(request, 'dashboard/doubts/my_bookmarks.html', {
        'doubts': page_obj,
        'is_bookmarked': {doubt.id: True for doubt in doubts}
    })

@login_required
def get_comments(request, doubt_id):
    doubt = get_object_or_404(Doubt, id=doubt_id)
    comments = doubt.comments.all().order_by('created_at')
    return render(request, 'dashboard/doubts/comments_list.html', {'comments': comments})


@login_required
def notifications_view(request):
    notifications = Notification.objects.filter(user=request.user).order_by('-created_at')
    
    # Mark notifications as read
    notifications.update(is_read=True)
    
    return render(request, 'dashboard/doubts/notifications.html', {'notifications': notifications})