from django.core.management.base import BaseCommand
from django.utils import timezone
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.conf import settings
from subscription.models import (
    AnnualSubscription,
    SubjectBasedSubscription,
    BundleSubscription
)

class Command(BaseCommand):
    help = 'Checks for expiring subscriptions and sends notifications'

    def handle(self, *args, **options):
        today = timezone.now().date()
        
        # Check subscriptions expiring in 7 days
        self.check_expiring_soon(today)
        
        # Check expired subscriptions
        self.check_expired(today)

    def check_expiring_soon(self, today):
        # Annual subscriptions expiring in 7 days
        annual_expiring = AnnualSubscription.objects.filter(
            end_date=today + timezone.timedelta(days=7),
            is_active=True
        )
        
        # Subject-based and bundle subscriptions expiring on Dec 31
        if today.month == 12 and today.day == 24:  # Check 7 days before Dec 31
            subject_expiring = SubjectBasedSubscription.objects.filter(
                end_date__year=today.year,
                is_active=True
            )
            bundle_expiring = BundleSubscription.objects.filter(
                end_date__year=today.year,
                is_active=True
            )
        else:
            subject_expiring = SubjectBasedSubscription.objects.none()
            bundle_expiring = BundleSubscription.objects.none()

        for subscription in list(annual_expiring) + list(subject_expiring) + list(bundle_expiring):
            self.send_expiry_notification(subscription, days_left=7)

    def check_expired(self, today):
        expired = AnnualSubscription.objects.filter(
            end_date=today,
            is_active=True
        ) | SubjectBasedSubscription.objects.filter(
            end_date=today,
            is_active=True
        ) | BundleSubscription.objects.filter(
            end_date=today,
            is_active=True
        )

        for subscription in expired:
            subscription.is_active = False
            subscription.save()
            self.send_expired_notification(subscription)

    def send_expiry_notification(self, subscription, days_left):
        subject = f'Your subscription expires in {days_left} days'
        message = render_to_string('emails/subscription_expiring.html', {
            'subscription': subscription,
            'days_left': days_left
        })
        send_mail(
            subject,
            message,
            settings.DEFAULT_FROM_EMAIL,
            [subscription.student.user.email],
            fail_silently=False,
            html_message=message
        )

    def send_expired_notification(self, subscription):
        subject = 'Your subscription has expired'
        message = render_to_string('emails/subscription_expired.html', {
            'subscription': subscription
        })
        send_mail(
            subject,
            message,
            settings.DEFAULT_FROM_EMAIL,
            [subscription.student.user.email],
            fail_silently=False,
            html_message=message
        )